from __future__ import annotations

import ctypes
from typing import Any, Dict, Tuple

try:
    import winreg  # type: ignore
except ImportError:  # pragma: no cover - only available on Windows
    winreg = None


def _set_registry_value(
    hive: int,
    path: str,
    name: str,
    value: Any,
    value_type: int,
) -> Tuple[bool, str]:
    if not winreg:
        return False, "winreg not available on this platform."
    try:
        with winreg.CreateKeyEx(hive, path, 0, winreg.KEY_SET_VALUE) as key:
            winreg.SetValueEx(key, name, 0, value_type, value)
        return True, "success"
    except PermissionError:
        return False, "Permission denied. Please run as administrator."
    except OSError as exc:
        return False, str(exc)


def apply_visual_effects_minimal() -> Tuple[bool, str]:
    """
    Configure Windows to use best performance visual effects.
    """
    settings = [
        (
            winreg.HKEY_CURRENT_USER,
            r"Software\Microsoft\Windows\CurrentVersion\Explorer\VisualEffects",
            "VisualFXSetting",
            2,
            winreg.REG_DWORD,
        ),
        (
            winreg.HKEY_CURRENT_USER,
            r"Control Panel\Desktop",
            "DragFullWindows",
            "0",
            winreg.REG_SZ,
        ),
        (
            winreg.HKEY_CURRENT_USER,
            r"Control Panel\Desktop\WindowMetrics",
            "MinAnimate",
            "0",
            winreg.REG_SZ,
        ),
    ]

    for hive, path, name, value, value_type in settings:
        ok, message = _set_registry_value(hive, path, name, value, value_type)
        if not ok:
            return ok, message

    try:
        ctypes.windll.user32.SystemParametersInfoW(0x0014, 0, None, 0)  # SPI_SETDESKWALLPAPER
    except Exception:
        pass

    return True, "Visual effects set to best performance."


def disable_background_apps() -> Tuple[bool, str]:
    """
    Prevent UWP background apps from running to reduce resource usage.
    """
    if not winreg:
        return False, "winreg not available."

    ok, message = _set_registry_value(
        winreg.HKEY_CURRENT_USER,
        r"Software\Microsoft\Windows\CurrentVersion\BackgroundAccessApplications",
        "GlobalUserDisabled",
        1,
        winreg.REG_DWORD,
    )
    if not ok:
        return ok, message

    ok, message = _set_registry_value(
        winreg.HKEY_CURRENT_USER,
        r"Software\Microsoft\Windows\CurrentVersion\Search",
        "BackgroundAppGlobalToggle",
        0,
        winreg.REG_DWORD,
    )
    if not ok:
        return ok, message

    return True, "Background apps disabled."


def disable_telemetry() -> Tuple[bool, str]:
    """
    Reduce telemetry level to minimum.
    """
    settings = [
        (
            winreg.HKEY_LOCAL_MACHINE,
            r"SOFTWARE\Policies\Microsoft\Windows\DataCollection",
            "AllowTelemetry",
            0,
            winreg.REG_DWORD,
        ),
        (
            winreg.HKEY_LOCAL_MACHINE,
            r"SOFTWARE\Policies\Microsoft\Windows\DataCollection",
            "AllowDeviceNameInTelemetry",
            0,
            winreg.REG_DWORD,
        ),
    ]

    for hive, path, name, value, value_type in settings:
        ok, message = _set_registry_value(hive, path, name, value, value_type)
        if not ok:
            return ok, message
    return True, "Telemetry set to minimum."

